//=============================================================================

#include <ctime>
#include <algorithm>
#include <iterator>
#include <vector>
#include "cinder/app/App.h"
#include "cinder/app/RendererGl.h"
#include "cinder/Color.h"
#include "cinder/gl/gl.h"
#include "cinder/Rand.h"
#include "cinder/Vector.h"
#include "ExplosionShip.h"
#include "Asteroid.h"
#include "Ship.h"
#include "Utils.h"

using namespace ci;
using namespace ci::app;
using namespace gl;
using namespace std;

//-----------------------------------------------------------------------------

class CinderApp : public App
{
public:
   static void prepareSettings(Settings* settings);
   void setup() override;
   void update() override;
   void draw() override;

   virtual void keyDown(KeyEvent event) override;
   virtual void keyUp(KeyEvent event) override;

private:
   void collisions();

   bool mKeyRight=false, mKeyLeft=false, mKeyUp=false, mKeyDown=false;
   bool mKeyFire = false;

   ShipPtr mShip;
   vector<TorpedoPtr> mTorpedos;
   vector<ExplosionPtr> mExplosions;
   vector<AsteroidPtr> mAsteroids;
};

//-----------------------------------------------------------------------------

void CinderApp::prepareSettings(Settings* settings)
{
   settings->setTitle("Cinder 19 - Schiff Asteroiden Kollisionen");
   settings->setResizable(false);
   settings->setWindowPos(100, 100);
   settings->setWindowSize(640, 480);
   settings->setFrameRate(30.0f);
}

//-----------------------------------------------------------------------------

void CinderApp::setup()
{
   randSeed(static_cast<unsigned long>(time(0)));

   Figure::sScreenWidth = static_cast<float>(getWindowWidth());
   Figure::sScreenHeigth = static_cast<float>(getWindowHeight());

   mShip = make_shared<Ship>();
   mAsteroids.push_back(make_shared<Asteroid>(Asteroid::Region::LeftTop, Asteroid::Type::Huge));
   mAsteroids.push_back(make_shared<Asteroid>(Asteroid::Region::RightTop, Asteroid::Type::Huge));
   mAsteroids.push_back(make_shared<Asteroid>(Asteroid::Region::LeftBottom, Asteroid::Type::Huge));
   mAsteroids.push_back(make_shared<Asteroid>(Asteroid::Region::RightBottom, Asteroid::Type::Huge));
}

//-----------------------------------------------------------------------------

void CinderApp::update()
{
   if (mKeyUp && mShip)
   {
      mShip->accelerate(+1.f);
   }
   if (mKeyDown && mShip)
   {
      mShip->accelerate(-1.f);
   }
   if (mKeyRight && mShip)
   {
      mShip->rotate(+9.f);
   }
   if (mKeyLeft && mShip)
   {
      mShip->rotate(-9.f);
   }

   if (mKeyFire && mShip)
   {
      TorpedoPtr f = mShip->fire();
      mTorpedos.push_back(f);
   }

   if (mShip)
   {
      mShip->update();
   }
   erase_if_c(mTorpedos, [](const TorpedoPtr& p) { return !p->update(); });
   erase_if_c(mExplosions, [](const ExplosionPtr& p) { return !p->update(); });
   erase_if_c(mAsteroids, [](const AsteroidPtr& p) { return !p->update(); });
   collisions();
}

//-----------------------------------------------------------------------------

void CinderApp::draw()
{
   clear(Color(0, 0, 0));

   if (mShip)
   {
      mShip->draw();
   }
   for_each_c(mTorpedos, [](const TorpedoPtr& p) { p->draw(); });
   for_each_c(mExplosions, [](const ExplosionPtr& p) { p->draw(); });
   for_each_c(mAsteroids, [](const AsteroidPtr& p) { p->draw(); });
}

//-----------------------------------------------------------------------------

void CinderApp::keyDown(KeyEvent event)
{
   switch (event.getCode())
   {
   case KeyEvent::KEY_RIGHT:
      mKeyRight = true;
      break;
   case KeyEvent::KEY_LEFT:
      mKeyLeft = true;
      break;
   case KeyEvent::KEY_UP:
      mKeyUp = true;
      break;
   case KeyEvent::KEY_DOWN:
      mKeyDown = true;
      break;
   case KeyEvent::KEY_x:
      mKeyFire = true;
      break;
   }
}

//-----------------------------------------------------------------------------

void CinderApp::keyUp(KeyEvent event)
{
   switch (event.getCode())
   {
   case KeyEvent::KEY_RIGHT:
      mKeyRight = false;
      break;
   case KeyEvent::KEY_LEFT:
      mKeyLeft = false;
      break;
   case KeyEvent::KEY_UP:
      mKeyUp = false;
      break;
   case KeyEvent::KEY_DOWN:
      mKeyDown = false;
      break;
   case KeyEvent::KEY_x:
      mKeyFire = false;
      break;
   }
}

//-----------------------------------------------------------------------------

void CinderApp::collisions()
{
   vector<TorpedoPtr> torpedoRemovables;
   vector<AsteroidPtr> asteroidRemovables;
   vector<ExplosionPtr> explosionNews;
   bool removeShip = false;

   // Kollision zwischen Schiff und Torpedo ("friendly (self) fire")
   if (mShip)
   {
      for_each_c(mTorpedos, [&](const TorpedoPtr& p)
      {
         if (distance(mShip->getLocation(), p->getLocation())<(Ship::Radius()+Torpedo::Radius()))
         {
            explosionNews.push_back(make_shared<ExplosionShip>(p->getLocation()));
            torpedoRemovables.push_back(p);
            removeShip = true;
         }
      });
   }

   // Kollision zwischen Schiff und Asteroid (nicht aufgepasst...)
   if (mShip)
   {
      for_each_c(mAsteroids, [&](const AsteroidPtr& p)
      {
         if (distance(mShip->getLocation(), p->getLocation()) < (Ship::Radius() + p->radius()))
         {
            explosionNews.push_back(make_shared<ExplosionShip>(mShip->getLocation()));
            asteroidRemovables.push_back(p);
            removeShip = true;
         }
      });
   }

   // Ergebnisse der Kollisionen festschreiben
   // - Torpedos entfernen
   // - Asteroiden entfernen
   // - Neue Explosionen uebernehmen
   // - Schiff zerstoeren, wenn notwendig
   for_each_c(torpedoRemovables, [this](const TorpedoPtr& p) { erase_c(mTorpedos, p); });
   for_each_c(asteroidRemovables, [this](const AsteroidPtr& p) { erase_c(mAsteroids, p); });
   copy_c(explosionNews, back_inserter(mExplosions));
   if (removeShip)
   {
      mShip.reset();
   }
}

//-----------------------------------------------------------------------------

CINDER_APP(CinderApp, RendererGl(), &CinderApp::prepareSettings)
