//=============================================================================

#include <algorithm>
#include "cinder/app/App.h"
#include "cinder/app/RendererGl.h"
#include "cinder/Color.h"
#include "cinder/gl/gl.h"
#include "cinder/Vector.h"
#include "Ship.h"
#include "ExplosionAsteroid.h"
#include "ExplosionShip.h"

using namespace ci;
using namespace ci::app;
using namespace gl;
using namespace std;

//-----------------------------------------------------------------------------

class CinderApp : public App
{
public:
   static void prepareSettings(Settings* settings);
   void setup() override;
   void update() override;
   void draw() override;

   virtual void keyDown(KeyEvent event) override;
   virtual void keyUp(KeyEvent event) override;

private:
   bool mKeyRight=false, mKeyLeft=false, mKeyUp=false, mKeyDown=false;
   bool mKeyFire = false;

   ShipPtr mShip;
   vector<TorpedoPtr> mTorpedos;
   vector<ExplosionPtr> mExplosions;
};

//-----------------------------------------------------------------------------

void CinderApp::prepareSettings(Settings* settings)
{
   settings->setTitle("Cinder 15 - Explosions Refactoring");
   settings->setResizable(false);
   settings->setWindowPos(100, 100);
   settings->setWindowSize(640, 480);
   settings->setFrameRate(30.0f);
}

//-----------------------------------------------------------------------------

void CinderApp::setup()
{
   Figure::sScreenWidth = static_cast<float>(getWindowWidth());
   Figure::sScreenHeigth = static_cast<float>(getWindowHeight());

   mShip = make_shared<Ship>();
}

//-----------------------------------------------------------------------------

void CinderApp::update()
{
   if (mKeyUp && mShip)
   {
      mShip->accelerate(+1.f);
   }
   if (mKeyDown && mShip)
   {
      mShip->accelerate(-1.f);
   }
   if (mKeyRight && mShip)
   {
      mShip->rotate(+9.f);
   }
   if (mKeyLeft && mShip)
   {
      mShip->rotate(-9.f);
   }

   if (mKeyFire && mShip)
   {
      //TorpedoPtr f = mShip->fire();
      //mTorpedos.push_back(f);
      ExplosionPtr e1 = make_shared<ExplosionShip>(vec2(150.f, 150.f));
      ExplosionPtr e2 = make_shared<ExplosionAsteroid>(vec2(250.f, 150.f));
      mExplosions.push_back(e1);
      mExplosions.push_back(e2);
   }

   if (mShip)
   {
      mShip->update();
   }
   auto eit = remove_if(begin(mTorpedos), end(mTorpedos), [](const TorpedoPtr& p) { return !p->update(); });
   mTorpedos.erase(eit, end(mTorpedos));
   auto eit2 = remove_if(begin(mExplosions), end(mExplosions), [](const ExplosionPtr& p) { return !p->update(); });
   mExplosions.erase(eit2, end(mExplosions));
}

//-----------------------------------------------------------------------------

void CinderApp::draw()
{
   clear(Color(0, 0, 0));

   if (mShip)
   {
      mShip->draw();
   }
   for_each(cbegin(mTorpedos), cend(mTorpedos), [](const TorpedoPtr& p) { p->draw(); });
   for_each(cbegin(mExplosions), cend(mExplosions), [](const ExplosionPtr& p) { p->draw(); });
}

//-----------------------------------------------------------------------------

void CinderApp::keyDown(KeyEvent event)
{
   switch (event.getCode())
   {
   case KeyEvent::KEY_RIGHT:
      mKeyRight = true;
      break;
   case KeyEvent::KEY_LEFT:
      mKeyLeft = true;
      break;
   case KeyEvent::KEY_UP:
      mKeyUp = true;
      break;
   case KeyEvent::KEY_DOWN:
      mKeyDown = true;
      break;
   case KeyEvent::KEY_x:
      mKeyFire = true;
      break;
   }
}

//-----------------------------------------------------------------------------

void CinderApp::keyUp(KeyEvent event)
{
   switch (event.getCode())
   {
   case KeyEvent::KEY_RIGHT:
      mKeyRight = false;
      break;
   case KeyEvent::KEY_LEFT:
      mKeyLeft = false;
      break;
   case KeyEvent::KEY_UP:
      mKeyUp = false;
      break;
   case KeyEvent::KEY_DOWN:
      mKeyDown = false;
      break;
   case KeyEvent::KEY_x:
      mKeyFire = false;
      break;
   }
}

//-----------------------------------------------------------------------------

CINDER_APP(CinderApp, RendererGl(), &CinderApp::prepareSettings)
